#include "Light.h"


Light::Light(int id, std::vector<GLfloat> pos)
{
	_ID = id;
	_Position = pos;
	_IsOn = true;
}


Light::~Light()
{
}

void Light::SetAttenuation(float constant, float linear, float quadratic)
{
	_ConstAtten = constant;
	_LinearAtten = linear;
	_QuadraticAtten = quadratic;
}

void Light::Update(float dt)
{
	if (_ShouldFlash)	//If we should flash
	{
		_DurationFlashed += dt;	//Increment by delta time

		//If we've gone past the duration
		if (_DurationFlashed >= _FlashDuration)
		{
			//Don't flash anymore, reset values
			_ShouldFlash = false;
			_FlashDuration = 0.0;
			_DurationFlashed = 0.0;
			Ambient = _PreFlashAmbient;
			Diffuse = _PreFlashDiffuse;
			CurrentlyFlashing = false;
		}

		else if (shouldAdd)	//If we should add the increment values, do so
		{
			Ambient = { Ambient[0] + (_FlashAmbValues[0] * dt), Ambient[1] + (_FlashAmbValues[1] * dt), Ambient[2] + (_FlashAmbValues[2] * dt) };
			Diffuse = { Diffuse[0] + (_FlashDifValues[0] * dt), Diffuse[1] + (_FlashDifValues[1] * dt), Diffuse[2] + (_FlashDifValues[2] * dt) };

			if (Ambient[0] >= _FlashAmbMax && Diffuse[0] >= _FlashDifMax)	//If we've hit or gone past the maximum values we should reach during flashing
			{
				shouldAdd = false;	//We should begin subtracting the increment value
			}
		}

		else if (!shouldAdd)	//If we should subtract the increment values, do so
		{
			Ambient = { Ambient[0] - (_FlashAmbValues[0] * dt), Ambient[1] - (_FlashAmbValues[1] * dt), Ambient[2] - (_FlashAmbValues[2] * dt) };
			Diffuse = { Diffuse[0] - (_FlashDifValues[0] * dt), Diffuse[1] - (_FlashDifValues[1] * dt), Diffuse[2] - (_FlashDifValues[2] * dt) };

			if (Ambient[0] <= 0.0f && Diffuse[0] <= 0.0f)	//If we get to or below 0
			{
				shouldAdd = true;	//Begin adding the increment values again
			}
		}
	}
}

void Light::Render()
{
	//Will be overridden
}

void Light::Flash(float cycleTime, float flashDuration, float dt)
{
	cycleTime = cycleTime * 10.0f;	//
	flashDuration = flashDuration * 10.0f;	// Times by 10 to convert to native units

	/*
		We use Ambient and Diffuse 0 to detect whether or not we need to increase or decrease intensity of the light during the flash
		If either are 0, we set them to a very tiny value (0.01) so the flash logic can work, but this is such a small value it makes no visible
		difference to the light
	*/

	if (Ambient[0] <= 0.0f)
	{
		Ambient[0] = 0.01f;
	}

	if (Diffuse[0] <= 0.0f)
	{
		Diffuse[0] = 0.01f;
	}

	if (!_ShouldFlash)	//If we're not already flashing
	{
		//Calculate Ambient and Diffuse increment values
		_FlashAmbValues = { Ambient[0] / (cycleTime / 2.0f), Ambient[1] / (cycleTime / 2.0f), Ambient[2] / (cycleTime / 2.0f) };
		_FlashDifValues = { Diffuse[0] / (cycleTime / 2.0f), Diffuse[1] / (cycleTime / 2.0f), Diffuse[2] / (cycleTime / 2.0f) };

		//Set maximum values Ambient and Diffuse should reach during flashing
		_FlashAmbMax = Ambient[0];
		_FlashDifMax = Diffuse[0];
		
		//We should begin flashing now, set the pre flash values
		_ShouldFlash = true;
		_FlashDuration = flashDuration;
		_PreFlashAmbient = Ambient;
		_PreFlashDiffuse = Diffuse;
		CurrentlyFlashing = true;
	}
}

void Light::SetAmbient(float r, float g, float b)
{
	Ambient = { r, g, b, 1.0f };
}

void Light::SetDiffuse(float r, float g, float b)
{
	Diffuse = { r, g, b, 1.0f };
}

int Light::GetID()
{
	return _ID;
}

bool Light::GetIsOn()
{
	return _IsOn;
}

void Light::SetIsOn(bool state)
{
	_IsOn = state;
}

std::vector<GLfloat> Light::GetPosition()
{
	return _Position;
}